--------------------------------------------------------------------------------------------
-- Symmetrical CA Control Panel   symCACP  cp version 
-- Create patterns from seeds run and step
-- Paul Rendell 14/10/2018
-- Some routines from sample script browse-patterns.lua written by Chris Rowett
-- 02/03/2023  reworked using a lot of code from 3D.lua Author: Andrew Trevorrow (andrew@trevorrow.com),
--------------------------------------------------------------------------------------------
-- Keyboard shortcuts:
--   Page Down  - next pattern
--   Page Up    - previous pattern
--   Home       - select a new folder to browse
--   O          - toggle options display
--   Control-I  - toggle pattern information display
--   Control-T  - toggle autofit
--   Esc        - exit at current pattern
--------------------------------------------------------------------------------------------
local title = 'SymCACP'
local build = 1   -- build number
local GlobalSwitch_ChaosSearch = "OSC"   --  "BREAK", "CHAOS", "OSC"
local GEOlabel = {['O'] = "Orhgthoganal+", ['D'] = "Diaganol      +", ['R'] = "Random        +", ['C'] = "CheckerBoard+",
                  ['I'] = "Island=20   +" }
local g = golly()
local gp = require "gplus"
local floor = math.floor
local op = require "oplus"
local gr = require("buildUni") 
local gs = require("search")
local gsChaos = require("search-chaos")
local gsCBrk = require("search-band")
local gf = require("script2") 
local ov = g.overlay

local modules = {}			-- optional modules like "script2-island"
local f = {}				-- local functions 
local util ={}				-- functions shared with scripting modules

local viewwd, viewht = g.getview(g.getlayer())
local overlaycreated = false

local    makeCookie  -- function

-- settings are saved in this file
local settingsFile = "SymCACP.ini"
local settingLst = {}		    -- list of setting names with the menu item numbers that have values and need saving
local mi = {}			    -- storage for menu items
local mbar                          -- the menu bar
local mbarht = 28                   -- height of menu bar
local ovwd, ovht                    -- current size of overlay
local ffTypeMenu                    -- pop-up menu for choosing a Find Fate Type

-- gui buttons
local menu = 2
local ruleFMTs = {'Hex', 'B/S'}
local ruleFMT = 1

local colonList, equalList
local m_Item = { button = {}, x = {}, y = {}, hiLight = {}, text = {}, menuNo = {}, textOnly = {}, bgColour={}, textColour={},
                     maxLabel = {}, func = {}, value = {}, Menu = {}, Item = {}, dots = {}}
local m_Items = 0
local ruleDecription = {}

---------------------------------------------------------------------------

local menuFile = 1
local menuEdit = 2
local menuUniverse = 3
local menuLists = 4
local menuView = 5
local menuLayer = 6
local menuTools = 7
local menuHelp = 8


---------------------------------------------------------------------------
-- TextType
local typeRule     = 'Hex or B/S'
local typeSeed     = 'Seed'
local typeRuleList = 'Rule List'
local typeSeedList = 'Seed List'
local typePositive = 'Positive'
local typeText     = 'Text'

-- position
local guiht        = 32          -- height of toolbar
local optht        = 0           -- height of options panel
local gapx         = 10          -- horitzonal gap between controls
local gapy         = 4           -- vertical gap between controls
local textrect     = "textrect"  -- clip name for clipping info text

-- style
local toolbarbgcolor    = "rgba 10 0 0 192"
local buttonBGhiLight   = "rgba 20 164 25 255" --"40 128 255 255" --op.buttonrgba --"20 164 25 255"
local buttonBGcolour    = "rgba 40 128 255 255"
local textBGcolour      = "rgba 255 255 255 255"
local helpBGcolour      = "rgba 255 253 217 255" -- pale yellow (matches Help window)
local transparantColour = "rgba 0 0 0 0"
local symSeedBGcolour   = "rgba 55 210 255 200"

-- for undo/redo
local undostack = {}                -- stack of states that can be undone
local redostack = {}                -- stack of states that can be redone
local startcount = 0                -- starting gencount (can be > 0)
local startstate = {}               -- starting state (used by Reset)
local dirty = false                 -- pattern has been modified?
local currentState = {}		    -- prestored state of everything

-- flags
local exitnow      = false  -- whether to exit
local islandResultsValid= false

-- log file
local scriptFile
local logFileName  = "logs\\SymCACPa.log"
local logFile = io.open ( logFileName , "w")
logFile:write("SymCACP.log\n\n")
logFile:flush()

local seed = '0'
local geo = ' '

local txtQHead		-- CMD input text buffer Queue Head
local txtQTail		-- CMD input text buffer Queue Tail

--==============================================================================
-- Utilites
--==============================================================================
--==================================================================
function util.GobalSwitch(switch, value)	-- used by script instruction
   if(switch == "CHAOS-SRCH") then
      GlobalSwitch_ChaosSearch = value
   end
end
--==================================================================
function util.file_exists(file)
  -- some error codes:
  -- 13 : EACCES - Permission denied
  -- 17 : EEXIST - File exists
  -- 20	: ENOTDIR - Not a directory
  -- 21	: EISDIR - Is a directory
  --
  local isok, errstr, errcode = os.rename(file, file)
  if isok == nil then
     if errcode == 13 then 
        -- Permission denied, but it exists
        return true
     end
     return false
  end
  return true
end
--------------------------------------------------------------------------------
 
function util.dir_exists(path)
  return util.file_exists(path .. "/")
end
--------------------------------------------------------------------------------

function util.TableConcat(t1,t2)
   for i=1,#t2 do
      t1[#t1+1] = t2[i]
   end
   return t1
end
--------------------------------------------------------------------------------
local function getString(prompt,value,title)
   local result = 'Q'
   function gString()
      result = g.getstring(prompt,value,title)
   end
   local status, err = pcall(gString)
   if err then
      g.continue("")  -- don't show error when script finishes
   end
   return(result)
end
--------------------------------------------------------------------------------
function util.note(text)
   function gnote()
      g.note(text)
   end
   local status, err = pcall(gnote)
   if err then
      g.continue("")  -- don't show error when script finishes
   end
end
--------------------------------------------------------------------------------
local function Unpack(t,p1,p2)
   newT = {}
   for i=p1,math.min(#t,p2) do
      newT[#newT+1] = t[i]
   end
   return newT
end
--------------------------------------------------------------------------------
local function spaces(n)
   return (' '):rep(n)
end
--------------------------------------------------------------------------------
function util.makeList(str)
   local list = {}
   for word in string.gmatch(str, "[^%s,]+") do
      list[#list + 1] = word
   end
   return list
end
--------------------------------------------------------------------------------
local function Strip_Control_Codes( str )
    local s = ""
    for i in str:gmatch( "%C+" ) do
 	s = s .. i
    end
    return s
end
--------------------------------------------------------------------------------
local function makeListLines(str)
   local list = {}
   for word in string.gmatch(str, "([^\n]*)\n?") do
      list[#list + 1] = word
   end
   return list
end
--------------------------------------------------------------------------------
function util.makeString(t1,coma)  		-- make a string from array elements
   local str = ''
   local delim = ''
   for i = 1,#t1 do
      str = str..delim..t1[i]
      delim = coma
   end
   return str
end
--------------------------------------------------------------------------------
local function makeStringSet(t1,coma)  		-- make a string from a set elements
   local str = ''
   local delim = ''
   local lst = {}
   for i,v in pairs(t1) do
      lst[#lst+1] = i
   end
   table.sort(lst)
   for i,v in ipairs(lst) do
      str = str..delim..v
      delim = coma
   end
   return str
end
--------------------------------------------------------------------------------
local function join(t1,t2)
   local t = {}
   for i = 1,#t1 do
      t[i] = t1[i]
   end
   for i = 1,#t2 do
      t[#t+1] = t2[i]
   end
   return t
end
--------------------------------------------------------------------------------
local function setAddList(set,list)		-- and items in list as inedex
   for i = 1,#list do
      set[list[i]] = true
   end
   return set
end
--------------------------------------------------------------------------------
local function setConCat(set,delim)		-- return concatenated string of a set indexes
   local str = ''
   local coma = ''
   list = {}
   for k, v in pairs(set) do
      list[#list+1] = k
   end
   table.sort(list)
   for i = 1, #list do
      str = str..coma..list[i]
      coma = delim
   end
   return str
end
--------------------------------------------------------------------------------
function util.reFormat(str, width)
   local result = '';
   local line = '';
   local space = '';
   local newLine = '';
   for word in (str..' '):gmatch("(.-)"..' ') do
      if (#line + #word) > width then
         result = result..newLine..line
         newLine = '\n'
         line = word
         space = ' '
      else
         line = line..space..word
         space = ' '
      end
   end
   if #line > 0 then
      result = result..newLine..line
   end
   return result
end
--------------------------------------------------------------------------------
function util.bool2txt(b)
   if b then
      return 'true'
   end
   return 'false'
end
--------------------------------------------------------------------------------
function util.strCC(s1,s2)
   local res = ''
   if  s2 then
      if s1 then
         res = s1..s2
      else
         res = '**NIL**'..s2
      end
   elseif s1 then
      res = s1..'**NIL**'
   else
      res = '**NIL*NIL**'
   end
   return res
end
--------------------------------------------------------------------------------
function util.zMatch(s1,s2)
   return util.strCC(s1,'z'):match(s2)
end
--------------------------------------------------------------------------------
function util.txtQAdd(text)
   lines = makeListLines(text)
   for i = 1,#lines do
      local el = {txt = Strip_Control_Codes(lines[i])}
      if (txtQTail) then
         txtQTail.next = el
      else
         txtQHead = el
      end
      txtQTail = el
   end
end
--------------------------------------------------------------------------------
function util.txtQRm()
   local txt,n
   if (txtQHead) then
      txt = txtQHead.txt
      n = txtQHead.n
      txtQHead = txtQHead.next
      if (not txtQHead) then
         txtQTail = nil
      end
   end
   return(txt)
end

-------------------------------------------------------------------------------
local function validRuleList(txt)
   res = true
   for rule in string.gmatch(txt, "[^%s,]+") do
      if not (string.find(rule, "^H%x*$") or string.find(rule, "^B%d*$") or string.find(rule, "^B%d*/S%d*$")) then
         res = false
         break
      end
   end
   return(res)
end
-------------------------------------------------------------------------------
--==============================================================================
local function reporter()
   local r={}
   function newSeg(segNo)
      r[segNo] = {}
      r[segNo].delim = ''
      r[segNo].result = true
      r[segNo].str = ''
   end
   function r.collect(str, res, segNo)
      segNo = segNo or 1
      if not r[segNo] then newSeg(segNo) end
      r[segNo].result = r[segNo].result and res
      r[segNo].str = r[segNo].str..r[segNo].delim..str
      r[segNo].delim = '\n'
   end
   function r.getReport(clear, segNo)
      segNo = segNo or 1
      if not r[segNo] then newSeg(segNo) end
      local str = r[segNo].str
      if clear then
         newSeg(segNo)
      end
      return str
   end
   function r.getResult(segNo)
      segNo = segNo or 1
      if not r[segNo] then newSeg(segNo) end
      return r[segNo].result
   end
   return r
end
--------------------------------------------------------------------------------

local function makeButton(bt, x,y,menuNo, text)
    m_Item.button[bt] = op.button(text, m_Item.func[bt])
    m_Item.x[bt] = x
    m_Item.y[bt] = y
    m_Item.menuNo[bt] = menuNo
    m_Item.bgColour[bt]= buttonBGcolour
    m_Item.textColour[bt]= op.textrgba
    return x + m_Item.button[bt].wd + gapx
end
--------------------------------------------------------------------------------

local function newMbarItem(text, value, doFunc)
    m_Items = m_Items +1
    m_Item.button[m_Items] = nil
    m_Item.func[m_Items] = doFunc
    m_Item.value[m_Items] = value
    m_Item.text[m_Items] = text
    if ( value ~= "") then 				-- has a value so needs saving and dots in text
       settingLst[text] = m_Items
       m_Item.dots[m_Items] = true
    end
    return m_Items
end
--------------------------------------------------------------------------------

local function m_ItemTxt(button)
   local text = m_Item.text[button]..m_Item.value[button]
   if (m_Item.dots[button]) then
      text = text.." ..."
   end
   return(text)
end
--------------------------------------------------------------------------------

local function addMbarLine(menu, item)
   mbar.additem(menu, "---", nil)
   return(item + 1)
end
--------------------------------------------------------------------------------

local function addMbarItem(menu, item, button)
   m_Item.Menu[button] = menu
   m_Item.Item[button] = item
   local text = m_Item.text[button]
   if (m_Item.value[button]) then text = m_ItemTxt(button) end
   mbar.additem(menu, text,  m_Item.func[button]);
   return(item + 1)
end
--------------------------------------------------------------------------------

local function reSetMbarButtonItem(button)
   mbar.setitem(m_Item.Menu[button], m_Item.Item[button], m_ItemTxt(button));
end
--------------------------------------------------------------------------------

local function enableMbarButtonItem(button, bool)
   mbar.enableitem(m_Item.Menu[button], m_Item.Item[button], bool);
   if (m_Item.button[button]) then
       m_Item.button[button].enable(bool)
   end
end
--------------------------------------------------------------------------------

local function showButton(bi)
    if m_Item.button[bi] and ((menu == 2) or (m_Item.menuNo[bi] == 1)) then
        local butRbga = op.buttonrgba
        op.textrgba = m_Item.textColour[bi]
        local shadow = {op.textshadowx,op.textshadowy}
        if m_Item.hiLight[bi] then
           op.buttonrgba = buttonBGhiLight
        elseif m_Item.textOnly[bi] then
           op.buttonrgba = toolbarbgcolor
        else
           op.buttonrgba = m_Item.bgColour[bi]
        end
        m_Item.button[bi].show(m_Item.x[bi],m_Item.y[bi])
        op.buttonrgba = butRbga
        op.textrgba = butTxtRbga
    end
end
--------------------------------------------------------------------------------

local function setlabel(bi,label)
    m_Item.value[bi] = label
    if (m_Item.dots[bi]) then
       if (mbar) then
          mbar.setitem(m_Item.Menu[bi], m_Item.Item[bi], m_ItemTxt(bi));
       end
    end
    if ( (bi <=  m_Items) and m_Item.button[bi]) then
       if label and (#label > 0) then
          local tempLabel = label
          if m_Item.maxLabel[bi] then
             if #label > m_Item.maxLabel[bi] then
                tempLabel = label:sub(1,m_Item.maxLabel[bi])..'...'
             end
          end
          m_Item.button[bi].setlabel(tempLabel, false)
       else
          m_Item.button[bi].setlabel(" ", false)
       end
       showButton(bi)
   end
end
--------------------------------------------------------------------------------

local function saveRule(rule)
   local ruleFormat = m_Item.value[mi.ruleFormat]:sub(1,1)
   if (rule:sub(1,1) ~= ruleFormat) then
      if ruleFormat == 'B' then
         rule = gr.convHex2BS(rule)
      else
         rule = gr.convBS2Hex(rule)
      end
   end
   local description = ruleDecription[gr.hexFormat(rule)]
   res = description ~= nil
   if (res) then
      setlabel(mi.rule, rule)
      setlabel(mi.ruleDesc, ruleDecription[gr.hexFormat(rule)])
   end
   return(res)
end
--------------------------------------------------------------------------------

local function setfont()
    op.textfont = "font 10 default-bold"
    if g.os() == "Mac" then
        op.yoffset = -1
    end
    if g.os() == "Linux" then
        op.textfont = "font 10 default"
    end
    ov(op.textfont)
end
--------------------------------------------------------------------------------

local function savesettings()
   local fw = io.open(settingsFile, "w")
   if fw then
      for name,bi  in pairs(settingLst) do
         fw:write(name.."="..m_Item.value[bi].."\n")
      end
      fw:close()
   else
      logFile:write("*** savesettings failed to open settingsfile\n")
   end
end
--------------------------------------------------------------------------------

local function loadsettings()
   local f = io.open(settingsFile, "r")
   if f then
      line = f:read("*line")
      while line do
         local name = line:match("[^=]*")
	 local value = line:match("[^=]*=(.*)")

         if (name) and (settingLst[name]) then
            if (not value) then value = " " end
            setlabel(settingLst[name], value)    
            if name == 'RULE' then
               saveRule(value)
            else
               if ((name == 'SEED')and gr.seedSym(value)) then
                  m_Item.bgColour[settingLst[name]] = symSeedBGcolour
               end
            end
         elseif (line:match("[=]")) then            
            logFile:write("*** loadsettings invalid entry "..line.."\n")            
         end
         line = f:read("*line")
      end
      f:close()
   else
      logFile:write("*** loadsettings failed to open settingsfile\n")
   end
end
--------------------------------------------------------------------------------

local function SaveState()
   local state = {}
   state.generation = g.getgen()	-- proxy for change in universe
   for name,bi  in pairs(settingLst) do
      state[name] = m_Item.value[bi]
   end
   state.uni = gr.GetUniverse()
   return(state)
end
--------------------------------------------------------------------------------

local function QueueState(state)
   redostack = {}
   undostack[#undostack+1] = state
end
--------------------------------------------------------------------------------

local function RestoreState(state)
   for name,bi  in pairs(settingLst) do
      m_Item.value[bi] = state[name]
   end
   gr.SetUniverse(state.uni)
   f.doFit()
end
--------------------------------------------------------------------------------

function f.SameState(state1, state2)
   local result = true
   for name,bi  in pairs(settingLst) do
      if (state1[name] ~= state2[name]) then
         result = false
         break
      end
   end
   return(result and state1.uni.isSame(state2.uni))
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
function f.checkStateChg(miIp)  -- called after a change so currentState is the old value is a change has occured
   local change
   local miList = {}
   if (miIp == nil) then
      change = true
   elseif (type(miIp) == 'table') then
      miList = miIp
   else
      miList = {miIp}
   end
   local c = " no Chg "; if change then c = " ChG!" end
   for ind,miI in pairs(miList) do
      if (miI == mi.gen) then 
         change = (currentState.generation ~= g.getgen())
      else
         change = (currentState[m_Item.text[miI]] ~= m_Item.value[miI])
      end
      if (change) then break end
   end
   if (change) then
      QueueState(currentState)
      currentState = SaveState()
     
   end
   for miI,ind in pairs(miList) do
      reSetMbarButtonItem(miI)
   end
   f.drawgui()  
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------

local function loadDescriptions()
   local descriptionsFile = m_Item.value[mi.ruleDescFile]
   local f = io.open(descriptionsFile, "r")
   local line = f:read("*line")
   while line do
      if line:upper():match "^%s*(.-)%s*\t" then
         local rule = gr.hexFormat(line:upper():match "^%s*(.-)%s*\t")
         ruleDecription[rule] = line:match "\t%s*(.-)%s*$"
         ruleDecription[gr.comp(rule)] = line:match "\t%s*(.-)%s*$":gsub('\t',' ')
      end
      line = f:read("*line")
   end
   f:close()
   saveRule(m_Item.value[mi.rule])   -- setup the text for description 
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------

function f.drawgui()
   -- draw toolbar background
   ov("blend 0")
   ov(toolbarbgcolor)      
   ov("fill 0 0 "..ovwd.." "..tBarht)

   -- draw main buttons
   f.EndableMBar()
   for bi =1, m_Items do
      showButton(bi)
   end
   g.update()
end

local function showhelpAB()
   g.open("helpAB.html")
end
--------------------------------------------------------------------------------

local function showhelp()
   g.open("help.html")
   showButton(mi.help)
end

local function doNoop()
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
local function getText(bi, prompt, textType)  -- textType { typeRule, typeSeed, typeRuleList, typePositive, typeText}
   local txt
   local thisPrompt = prompt
   local tryAgain = true
   local oldText = m_Item.value[bi]
   local butTxtRbga = op.textrgba
   
   while tryAgain do
      tryAgain = false
      txt = getString(thisPrompt, oldText, title):upper()
      if txt:find('[Qq]') then
         break
      end
      if (textType == typePositive) then
	 if (string.find(txt, "[^%d]")) then
	    thisPrompt = prompt..' invalid '..txt..' try again'
	    oldText = txt
	    tryAgain = true
	 end
      elseif (textType == typeRule) then
	 if (not (( string.find(txt, "^H%x*$") or string.find(txt, "^B%d*$") or string.find(txt, "^B%d*/S%d*$")) and saveRule(txt:upper()))) then
	    thisPrompt = prompt..' invalid '..txt..' try again'
	    oldText = txt
	    tryAgain = true
	 end
      elseif (textType == typeSeedList) then
         local seedno = 0
         txt = txt:lower()
	 m_Item.bgColour[bi] = buttonBGcolour 
	 local list = util.makeList(txt)
	 local symSeed = false
         for i=1,#list do
            local seed = list[i]
            seedno = seedno +1
            if not (string.find(seed, "^[0-9oabc]+$") or  string.find(seed, "^[0123]+$")) then
               thisPrompt = prompt..' seed '..seed..'('..string.len(seed)..') number '..seedno..' invalid '..txt..' try again'
               logFile:write(thisPrompt.."\n")
  	       oldText = txt
	       tryAgain = true
            end
            if (not tryAgain) and gr.seedSym(seed) then
               symSeed = true
   	       m_Item.bgColour[bi] = symSeedBGcolour
   	       g.show("symetric seed number "..seedno)
            end
         end
         if  (symSeed)  then
	    m_Item.bgColour[bi] = symSeedBGcolour
	    g.show("symetric seed")
	 else
	    m_Item.bgColour[bi] = buttonBGcolour 
	    g.show("")
	 end
      elseif (textType == typeSeed) then
         txt = txt:lower()
         if not (string.find(txt, "^[0-9oabc]+$") or string.find(txt, "^[0123]+$")) then
	    thisPrompt = prompt..' invalid '..txt..' try again'
	    g.show(thisPrompt)
	    oldText = txt
	    tryAgain = true
	 else
	    if (gr.seedSym(txt)) then
	       m_Item.bgColour[bi] = symSeedBGcolour
	       g.show("symetric seed")
	    else
	       m_Item.bgColour[bi] = buttonBGcolour 
	       g.show("")
	    end
	 end
      elseif (textType == typeRuleList) then
         if (not validRuleList(txt) ) then
	    thisPrompt = prompt..' invalid '..txt..' try again'
	    oldText = txt
	    tryAgain = true
	 end
      elseif (textType == typeText) then
         tryAgain = false
      end
   end
   if not txt:find('[Qq]') then
      if (bi ~= mi.rule) then
         if (bi == mi.seed) or (bi == mi.seedList) then
            setlabel(bi, txt:lower())
         else
            setlabel(bi, txt:upper())
         end
      end
   end
   return not txt:find('[Qq]') 
end
--------------------------------------------------------------------------------

local function updateRuleList()
   local listTxT = ""
   local ruleTxt
   g.show("updateRuleList:"..m_Item.value[mi.ruleList]..":")
   for rule in string.gmatch(m_Item.value[mi.ruleList], "[^%s,]+") do
      if ruleFMT < #ruleFMTs then  -- Hex
         ruleTxt = gr.convBS2Hex(rule)
      else			   -- B/S
         ruleTxt = gr.convHex2BS(rule)
      end
      if (listTxT ~= "") then
         listTxT = listTxT..","
      end
      listTxT = listTxT..ruleTxt
   end
   setlabel(mi.ruleList,listTxT)
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------

local function doLayer_delEx()			-- layer commands
   while ( g.numlayers() > 1 ) do
      g.setlayer(g.numlayers() - 1)
      g.dellayer()
   end
   enableMbarButtonItem(mi.layDelExtra, false)
   g.update()
end
--------------------------------------------------------------------------------

local function reset_StackTile(noNote)
   local val = "false"
   if (g.getoption("stacklayers")==1) then val = "true" end
   m_Item.value[mi.layStack] = val
   reSetMbarButtonItem(mi.layStack)
   g.update()
   val = "false"
   if (g.getoption("tilelayers")==1) then val = "true" end
   m_Item.value[mi.layTile] = val
   reSetMbarButtonItem(mi.layTile)
   g.update()
   if (not noNote) then
      util.note("stacklay "..g.getoption("stacklayers").." tile "..g.getoption("tilelayers"))
   end
end
--------------------------------------------------------------------------------

local function doLayer_Stack()			-- layer toggle stack
   g.setoption("stacklayers", 1 - g.getoption("stacklayers"))
  reset_StackTile()
end
--------------------------------------------------------------------------------

local function doLayer_Tile()			-- layer toggle tile
   g.setoption("tilelayers", 1 - g.getoption("tilelayers"))
   reset_StackTile()
end
--------------------------------------------------------------------------------

local function doLayer_Set()			-- layer toggle tile
   if (getText(mi.SetLayer, "Layer", typePositive))  then
      local lay = tonumber(m_Item.value[mi.SetLayer])
      if (g.numlayers() > lay) then
         g.setlayer(lay)
         reSetMbarButtonItem(mi.layTile)
         g.update()
      else
         util.note("Layer "..lay.." invalid")
      end
   end
end
--------------------------------------------------------------------------------

local function doRuleFormat()
   if ruleFMT < #ruleFMTs then  -- B/S
      ruleFMT = ruleFMT + 1
      setlabel(mi.rule, gr.convHex2BS(m_Item.value[mi.rule]))
   else				-- Hex
      ruleFMT = 1
      setlabel(mi.rule, gr.convBS2Hex(m_Item.value[mi.rule]))
   end
   setlabel(mi.ruleFormat,ruleFMTs[ruleFMT].." +")
   updateRuleList()
   f.checkStateChg({mi.rule, mi.ruleFormat})
   g.update()
end
--------------------------------------------------------------------------------

local function doStep()
   gr.doStep()
   f.checkStateChg({mi.gen})
end
--------------------------------------------------------------------------------

local function doRunN()
   setlabel(mi.run,"Stop")
   m_Item.hiLight[mi.run] = true
   showButton(mi.run)
   g.update()
   if (getText(mi.runN, "Enter Generation to run :", typePositive)) then
      gr.doStepN(tonumber(m_Item.value[mi.runN]))
      f.checkStateChg({mi.gen, mi.runN})
   end
   setlabel(mi.run, "Run")
   m_Item.hiLight[mi.run] = false
   showButton(mi.run)
   g.update()
end
--------------------------------------------------------------------------------

local function dorun2G()
   setlabel(mi.run,"Stop")
   m_Item.hiLight[mi.run] = true
   showButton(mi.run)
   g.update()
   if (getText(mi.run2G, "Enter Generation to run to:", typePositive)) then
      local step = tonumber(m_Item.value[mi.run2G]) - tonumber(g.getgen())
      if (step > 0) then
         gr.doStepN(step)
         f.checkStateChg({mi.gen, mi.run2G})
      end
   end
   setlabel(mi.run,"Run")
   m_Item.hiLight[mi.run] = false
   showButton(mi.run)
   g.update()
end
--------------------------------------------------------------------------------

local function doRun()
   setlabel(mi.run,"Stop")
   m_Item.hiLight[mi.run] = true
   showButton(mi.run)
   g.update()
   gr.doRun()
   setlabel(mi.run, "Run")
   m_Item.hiLight[mi.run] = false
   showButton(mi.run)
   g.update()
   f.checkStateChg(mi.gen)
end
--------------------------------------------------------------------------------

local function doFindFate()
   local ans="failed"
   m_Item.hiLight[mi.findFate] = true
   g.show("findFate started")
   if ((GlobalSwitch_ChaosSearch:sub(1,5) == "BREAK") or (GlobalSwitch_ChaosSearch == "CHAOS")) then  --BREAK, CHAOS, OSC, OLDCHAOS
      logFile:write("findFate_BRK: "..GlobalSwitch_ChaosSearch.."\n")
      gsCBrk.init(logFile, GlobalSwitch_ChaosSearch)
      ans = gsCBrk.doFindEnd(m_Item.value[mi.seed], tonumber(m_Item.value[mi.maxGen]), tonumber(m_Item.value[mi.FEstep1]),  tonumber(m_Item.value[mi.FEminOsc]), tonumber(m_Item.value[mi.FEmaxOsc]))		-- (seed, maxGen, step1, oscLenMin, oscLenMax)
   elseif (GlobalSwitch_ChaosSearch == "OLDCHAOS") then
      gsChaos.init(logFile)
      logFile:write("findFate "..GlobalSwitch_ChaosSearch.."\n")
      ans = gsChaos.doFindEnd(m_Item.value[mi.seed], tonumber(m_Item.value[mi.maxGen]), tonumber(m_Item.value[mi.FEstep1]),  tonumber(m_Item.value[mi.FEminOsc]), tonumber(m_Item.value[mi.FEmaxOsc]))		-- (seed, maxGen, step1, oscLenMin, oscLenMax)
   else
      logFile:write("findFate\n")
     ans = gs.doFindEnd(m_Item.value[mi.seed], tonumber(m_Item.value[mi.maxGen]), tonumber(m_Item.value[mi.FEstep1]),  tonumber(m_Item.value[mi.FEminOsc]), tonumber(m_Item.value[mi.FEmaxOsc]))		-- (seed, maxGen, step1, oscLenMin, oscLenMax)
   end
   m_Item.hiLight[mi.findFate] = false
   f.checkStateChg(mi.gen)
   return(ans)
end
--------------------------------------------------------------------------------

local function setFFosc() 
   GlobalSwitch_ChaosSearch = "OSC"
   m_Item.value[mi.findFateType] = GlobalSwitch_ChaosSearch
   reSetMbarButtonItem(mi.findFateType)
end
--------------------------------------------------------------------------------
local function setFFbreak() 
   GlobalSwitch_ChaosSearch = "BREAK"
   m_Item.value[mi.findFateType] = GlobalSwitch_ChaosSearch
   reSetMbarButtonItem(mi.findFateType)
end
--------------------------------------------------------------------------------

local function setFFchaos() 
   GlobalSwitch_ChaosSearch = "Chaos"
   m_Item.value[mi.findFateType] = GlobalSwitch_ChaosSearch
   reSetMbarButtonItem(mi.findFateType)
end
--------------------------------------------------------------------------------

local function setIsleTypeRand() 
   m_Item.value[mi.isleType] = "Random"
   reSetMbarButtonItem(mi.isleType)
end
--------------------------------------------------------------------------------

local function setIsleTypeSolid() 
   m_Item.value[mi.isleType] = "Solid"
   reSetMbarButtonItem(mi.isleType)
end
--------------------------------------------------------------------------------

local function resultsHeader()
   local width =   m_Item.value[mi.width]
   local hight =   m_Item.value[mi.hight]
   local text = "Island Analysis Results\nRule: "..m_Item.value[mi.rule]
   if (width == "0") then width = hight end
   if (hight == "0") then hight = width end
   text = text.." "..width.."*"..hight.." seed "..m_Item.value[mi.seed].." Gen "..g.getgen().."\n"
   return(text)
end
--------------------------------------------------------------------------------

local function doIsland()
   if ( not modules["island"]) then
      modules["island"] =  require("script2-island")
   end
   modules["island"].findIslands(logFile)
   modules["island"].ShowIslands()
   modules["island"].result = resultsHeader()..modules["island"].BigIslandStats()
   g.setoption("stacklayers",1)
   enableMbarButtonItem(mi.layDelExtra, g.numlayers() > 1)
   islandResultsValid= true
   util.note(modules["island"].result)
end
--------------------------------------------------------------------------------

local function doislandResults()
   if (modules["island"] and modules["island"].result) then
      util.note(modules["island"].result)
   else
      util.note("No Island Results")
   end
end
--------------------------------------------------------------------------------

local function doBuilding(bldType, miList)
   local width =   m_Item.value[mi.width]
   local hight =   m_Item.value[mi.hight]
   local seed =    m_Item.value[mi.seed]:gsub(' ','')
   local geo =     m_Item.value[mi.geo]
   local rule =    m_Item.value[mi.rule]
   if rule:sub(1,1) == 'B' then
      rule = gr.convBS2Hex(rule)
   end
   local fullGeo = geo
   if (geo:sub(1,1) == "I") then
      fullGeo = geo..m_Item.value[mi.geoSize].."%"..m_Item.value[mi.randomPC]
      if (m_Item.value[mi.isleType] == "Solid") then
         fullGeo = fullGeo.."S"
      end
   elseif(geo:sub(1,1):upper() == 'R') then
      fullGeo = geo.."%"..m_Item.value[mi.randomPC]
   end
   if (bldType == "Build") then
      GlobalSwitch_ChaosSearch = m_Item.value[mi.findFateType]
      gr.doBuild('SymCACP ',width, hight, seed, fullGeo, rule)  -- w, h, seed, rule, format
   else
      local rule=gr.doSwapComp('SymCACP ',width, hight, seed, fullGeo, rule)  -- w, h, seed, rule, format
      setlabel(mi.rule, rule)
   end
   f.doFit()
   f.checkStateChg(miList)
   g.update()
end
--------------------------------------------------------------------------------

local function doBuild(miList)
   doBuilding("Build", miList)
   showButton(mi.build)
end
--------------------------------------------------------------------------------

local function doSwapComp()
   m_Item.hiLight[mi.swapComp] = true
   showButton(mi.swapComp)
   g.update()
   doBuilding("SwapComp", mi.swapComp)
   m_Item.hiLight[mi.swapComp] = false
   showButton(mi.swapComp)
   g.update()
end

--------------------------------------------------------------------------------
local function doTextRuleList()
   if getText(mi.ruleList, 'Rule list', typeRuleList) then
      m_Item.value[mi.nextRule] = '1'
      m_Item.value[mi.ruleListNo] = m_Item.value[mi.nextRule]
      reSetMbarButtonItem(mi.nextRule)
      f.checkStateChg({mi.ruleList, mi.ruleListNo})
   end
end
--------------------------------------------------------------------------------
local function doTextRule()
   if getText(mi.rule, 'Rule', typeRule) then
      local rule = m_Item.value[mi.rule]
      local ruleFormat = m_Item.value[mi.ruleFormat]
      if (rule:find('^B') and not ruleFormat:find('^B')) or
         (rule:find('^H') and not ruleFormat:find('^H')) then
         ruleFMT = 3 - ruleFMT
         setlabel(mi.ruleFormat,ruleFMTs[ruleFMT].." +")
      end
      reSetMbarButtonItem(mi.rule)
      f.checkStateChg({mi.rule})
   end
   doBuild(mi.rule)
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
local function doRuleDesc()
-- showhelp
   util.note(util.reFormat("Rule "..m_Item.value[mi.rule].." "..m_Item.value[mi.ruleDesc],60))
   showButton(mi.ruleDesc)
end
--------------------------------------------------------------------------------
local function doTextWidth()
   getText(mi.width, 'Width', typePositive)
   reSetMbarButtonItem(mi.width)
   doBuild(mi.width)
end
--------------------------------------------------------------------------------
local function doTextHight()
   getText(mi.hight, 'Height', typePositive)
   reSetMbarButtonItem(mi.hight)
   doBuild(mi.hight)
end
--------------------------------------------------------------------------------
local function doTextSeed()
   getText(mi.seed, 'Seed', typeSeed)
   reSetMbarButtonItem(mi.seed)
   doBuild(mi.seed)
   showButton(mi.seed)
end
--------------------------------------------------------------------------------
local function doTextmaxGen()
   getText(mi.maxGen, 'Maximum Generation', typePositive)
   reSetMbarButtonItem(mi.maxGen)
   f.checkStateChg(mi.maxGen)
end
--------------------------------------------------------------------------------
local function doTextStep1Gen()
   getText(mi.FEstep1, 'Generation fist step', typePositive)
   reSetMbarButtonItem(mi.FEstep1)
   f.checkStateChg(mi.FEstep1)
end
--------------------------------------------------------------------------------
local function doTextMinOsc()
   getText(mi.FEminOsc, 'Minimum Oscillation Length', typePositive)
   reSetMbarButtonItem(mi.FEminOsc)
   f.checkStateChg(mi.FEminOsc)
end
--------------------------------------------------------------------------------
local function doTextMaxOsc()
   getText(mi.FEmaxOsc, 'Maximum Oscillation Length', typePositive)
   reSetMbarButtonItem(mi.FEmaxOsc)
   f.checkStateChg(mi.FEmaxOsc)
end
--------------------------------------------------------------------------------
local function doFEopsMenu()
    ffTypeMenu.show(290, 60, ovwd, viewht)
end
--------------------------------------------------------------------------------

local function exitbrowser()
   exitnow = true
end
--------------------------------------------------------------------------------

local function doGeometry()
   if (m_Item.value[mi.geo]:sub(1,1) == "D") then
      setlabel(mi.geo,GEOlabel['O'])
   elseif (m_Item.value[mi.geo]:sub(1,1) == "O") then
      setlabel(mi.geo,GEOlabel['C'])
   elseif (m_Item.value[mi.geo]:sub(1,1) == "C") then
      setlabel(mi.geo,GEOlabel['R'])
   else
      setlabel(mi.geo,GEOlabel['D'])
   end
   showButton(mi.geo)
   doBuild(mi.geo)
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------

local function checkIslandSize(geo)
   local txt = m_Item.value[mi.geoSize]
   local wd = tonumber(m_Item.value[mi.width])
   local ht = tonumber(m_Item.value[mi.hight])
   if (ht == 0) then 
      ht = wd
   elseif (wd == 0) then
      wd = ht
   end
   local thisPrompt = m_Item.value[mi.geoSize]
   local space = tonumber(m_Item.value[mi.geoSize])
   if (not space) then space = 0 end
   if (geo:sub(1,2) == 'ID') then space = space * 1.42 end
   local tryAgain = ((space+1)/2  >  math.min(ht,wd)//2) or (space < 1)
   while tryAgain do
      tryAgain = false
      txt = getString(thisPrompt, m_Item.value[mi.geoSize],
                        "Island Size takes "..space.." cells but only "..math.min(ht,wd)-1):upper()
      if txt:find('Q') then
         break
      elseif (string.find(txt, "[^%d]")) then
	 thisPrompt = prompt..txt..' is not a number try again'
	 tryAgain = true
      else
         tryAgain = (tonumber(txt)//2 > (math.min(ht,wd)+2)//2) or (tonumber(txt) < 1)
         thisPrompt = txt
      end
   end
   if (not txt:find('Q')) then
      m_Item.value[mi.geoSize] = txt
      reSetMbarButtonItem(mi.geoSize)
   end
   return (not txt:find('Q'))
end

--------------------------------------------------------------------------------
local function doGeoSize()
   getText(mi.geoSize, 'Island Size', typePositive)
   if ( checkIslandSize(m_Item.value[mi.geo])) then
      doBuild(mi.geoSize)
   end
end

--------------------------------------------------------------------------------
local function doRandomPC()
   if (getText(mi.randomPC, 'Random Percentage', typePositive)) then
      doBuild(mi.randomPC)
   end
end

--------------------------------------------------------------------------------
local function doIsleType()
   local oldType = m_Item.value[mi.isleType]
   isleTypeMenu.show(150, 250, ovwd, viewht)
   reSetMbarButtonItem(mi.isleType)
   if (oldType ~= m_Item.value[mi.isleType]) then
       doBuild(mi.isleType)
   end
end

--------------------------------------------------------------------------------
local function doInvertIsle()
   gr.Switch_InvertIsland = not gr.Switch_InvertIsland
   if (gr.Switch_InvertIsland) then
      m_Item.value[mi.invertIsle] = "true"
   else
      m_Item.value[mi.invertIsle] = "false"
   end
   reSetMbarButtonItem(mi.invertIsle)
   doBuild(mi.invertIsle)
end

--------------------------------------------------------------------------------

local function doGeoMenu()
   geoMenu.show(150, 250, ovwd, viewht)
   reSetMbarButtonItem(mi.geo)
end
--------------------------------------------------------------------------------

local function setGeoDiag()
   setlabel(mi.geo,'Diagonal')
   doBuild(mi.geo)
end
--------------------------------------------------------------------------------

local function setGeoRandDiag()
   setlabel(mi.geo,'Random_Diag')
   doBuild(mi.geo)
end

--------------------------------------------------------------------------------
local function setGeoOrthog()
   setlabel(mi.geo,'Orthogonal')
   doBuild(mi.geo)
end
--------------------------------------------------------------------------------
local function setGeoRand()
   setlabel(mi.geo,'Random')
   doBuild(mi.geo)
end
--------------------------------------------------------------------------------
local function setGeoCB()
   setlabel(mi.geo,'Checkerboard')
   doBuild(mi.geo)
end
--------------------------------------------------------------------------------
local function setGeoIsleR()
   if ( checkIslandSize('IR')) then
      setlabel(mi.geo,'IR-Round Island')
      doBuild(mi.geo)
   end
end
--------------------------------------------------------------------------------
local function setGeoIsleD()
   if ( checkIslandSize('D')) then
      setlabel(mi.geo,'ID-Diamond Island')
      doBuild(mi.geo)
   end
end
--------------------------------------------------------------------------------
local function setGeoIsleS()
   if ( checkIslandSize('IS')) then
      setlabel(mi.geo,'IS-Square Island')
      doBuild(mi.geo)
   end
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------

local function doBldnextRule()
   local ruleList = {}
   if ( m_Item.value[mi.nextRule] ~= "*") then
      local ruleNo = math.max(1, tonumber(m_Item.value[mi.nextRule]))
      for rule in string.gmatch(m_Item.value[mi.ruleList], "[^%s,]+") do
         ruleList[#ruleList + 1] = rule
      end
      if (ruleNo <= #ruleList) then
         saveRule(ruleList[ruleNo])
         ruleNo = ruleNo + 1
         if (ruleNo <= #ruleList) then
            m_Item.value[mi.nextRule] = tostring(ruleNo)
         else
            m_Item.value[mi.nextRule] = "*"
         end
         m_Item.value[mi.ruleListNo] = m_Item.value[mi.nextRule]
         doBuild({mi.rule, mi.ruleListNo, mi.nextRule})
         reSetMbarButtonItem(mi.nextRule)
         reSetMbarButtonItem(mi.ruleListNo)
      end   
   end   
   f.drawgui()   
end
--------------------------------------------------------------------------------
 
local function doChgRuleListNo()
   local try = true
   local maxNo = 0 
   local str=""
   for x in m_Item.value[mi.ruleList]:gmatch('[/HS0-9ABCDEF]+') do maxNo = maxNo+1;str=str..","..x end
   local prompt = 'Next Rule List No'
   while try do
      if getText(mi.nextRule, prompt, typePositive) then
         local rn = tonumber(m_Item.value[mi.nextRule])
         try =  (rn > maxNo ) or (rn < 1)
         g.show("rn "..rn.." maxNo "..maxNo.." str "..str)
         if try then
            prompt = '0 < Rule List No <= '..tostring(maxNo)
         else
            m_Item.value[mi.ruleListNo] = m_Item.value[mi.nextRule]
            reSetMbarButtonItem(mi.nextRule)
            reSetMbarButtonItem(mi.ruleListNo)
            f.checkStateChg({mi.nextRule, mi.ruleListNo})
         end
      else
         try = false
         m_Item.value[mi.nextRule] = currentState[m_Item.text[mi.nextRule]]
      end
   end
end

--------------------------------------------------------------------------------
-------------------------------------------------------------------------------
--------------------------------------------------------------------------------

local function doBldnextSeed()
   local seedList = {}
   if ( m_Item.value[mi.nextSeed] ~= "*") then
      local seedNo = math.max(1, tonumber(m_Item.value[mi.nextSeed]))
      for seed in string.gmatch(m_Item.value[mi.seedList], "[^%s,]+") do
         seedList[#seedList + 1] = seed
      end
      if (seedNo <= #seedList) then
         m_Item.value[mi.seed] = seedList[seedNo]
         seedNo = seedNo + 1
         if (seedNo <= #seedList) then
            m_Item.value[mi.nextSeed] = tostring(seedNo)
         else
            m_Item.value[mi.nextSeed] = "*"
         end
         m_Item.value[mi.seedListNo] = m_Item.value[mi.nextSeed]
         reSetMbarButtonItem(mi.nextSeed)
         reSetMbarButtonItem(mi.seedListNo)
         reSetMbarButtonItem(mi.seed)
         doBuild({mi.seed,mi.nextSeed,mi.seedListNo})
      end   
   end   
   f.drawgui()
end
--------------------------------------------------------------------------------
 
local function doChgSeedListNo()
   local try = true
   local maxNo = 0 
   local str=""
   for x in m_Item.value[mi.seedList]:gmatch('[/HS0-9ABCDEF]+') do maxNo = maxNo+1;str=str..","..x end
   local prompt = 'Next Seed List No'
   while try do
      if getText(mi.nextSeed, prompt, typePositive) then
         local rn = tonumber(m_Item.value[mi.nextSeed])
         try =  (rn > maxNo ) or (rn < 1)
         g.show("rn "..rn.." maxNo "..maxNo.." str "..str)
         if try then
            prompt = '0 < Seed List No <= '..tostring(maxNo)
         else
            m_Item.value[mi.seedListNo] = m_Item.value[mi.nextSeed]
            reSetMbarButtonItem(mi.nextSeed)
            reSetMbarButtonItem(mi.seedListNo)
            f.checkStateChg({mi.nextSeed, mi.seedListNo})
         end
      else
         try = false
         m_Item.value[mi.nextSeed] = currentState[m_Item.text[mi.nextSeed]]
      end
   end
end

--------------------------------------------------------------------------------
local function doTextSeedList()
   local oldVal = m_Item.value[mi.seedList]
   if (getText(mi.seedList, 'Seed list', typeSeedList)) then
      m_Item.value[mi.nextSeed] = '1'
      m_Item.value[mi.seedListNo] = m_Item.value[mi.nextSeed]
      f.checkStateChg({mi.nextSeed, mi.seedListNo})
      reSetMbarButtonItem(mi.nextSeed)
   end
end

--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
local function findPixelsPerCell(barHt, viewHt, wd, ht, mag)
   local ppc = (viewHt-barHt)/ht
   if (mag  < 5) then
      local selHt = ht
      local again =true
      while (again) do
         selHt = selHt - 1
         if (selHt > 1)  then
            g.select({-(wd//2),-(ht//2), 1, selHt})
            g.fitsel()
            again = (g.getmag() == mag)
         else
           again = false
         end
      end
      if (g.getmag() ~= mag) then
         ppc = viewHt/(selHt*2)
         g.setmag(mag)
      end
   end
   return(ppc)
end

--------------------------------------------------------------------------------
function f.doFit()
   local wd = tonumber(m_Item.value[mi.width])
   local ht = tonumber(m_Item.value[mi.hight])
   if (wd < 4) then wd = 4 end
   if (ht == 0) then ht = wd end
   local viewWd, viewHt = g.getview(g.getlayer())
   local barHt = (mbarht + tBarht)
   g.select({-(wd//2),-(ht//2),wd,ht})
   g.fitsel()
   local mag = g.getmag()
   local ppc = findPixelsPerCell(barHt, viewHt, wd, ht, mag)
   if (mag > 0) and ( ht*ppc > (viewHt - barHt)) then
   --  need to change magnification to fit in
      mag = mag-1
      ppc = ppc//2
      g.setmag(mag)
   end
   local selHt = math.max(1,math.floor(ht - barHt/ppc + 0.5))
   if (selHt < 3) then selHt = 3 end	-- ***** could try something better?
   g.select({0,-(ht//2), 1, selHt}) 
   g.fitsel()
   if( g.getmag() ~= mag) then
      g.setmag(mag)
   end   
   g.select({})
end

--------------------------------------------------------------------------------
local function doScriptList()
   g.show("calling script list")
   gf.doScriptList()
end
--------------------------------------------------------------------------------
   
local function doScriptLoad()
   gf.doScriptLoad()
end
--------------------------------------------------------------------------------

local function doScriptPaste()			-- layer toggle tile
   local txt = g.getclipstr()
   gf.ScriptLoad("Pasted", makeListLines(txt))
   enableMbarButtonItem(mi.sciptList,     gf.ScriptLoaded())
   enableMbarButtonItem(mi.sciptRun,      gf.ScriptLoaded())
end
--------------------------------------------------------------------------------
   
local function doScriptRun()
   m_Item.hiLight[mi.sciptRun] = true
   gf.doScriptRun(false)
   m_Item.hiLight[mi.sciptRun] = false
   showButton(mi.sciptRun)
   f.checkStateChg()
end
--------------------------------------------------------------------------------
   
local function doRulesDir()
   local ruleDir
   if (m_Item.value[mi.ruleDir]) then 
      ruleDir = m_Item.value[mi.ruleDir]
   else
      ruleDir = g.getdir("app")
   end
   ruleDir = g.opendialog('Folder for Odd Rules', "dir",ruleDir, "Rules", true)
   util.note(ruleDir)
   if (ruleDir)  then
      m_Item.value[mi.ruleDir] = ruleDir
      g.setdir("rules",m_Item.value[mi.ruleDir])
      f.checkStateChg(mi.ruleDir)
      reSetMbarButtonItem(mi.ruleDir)
   end
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
--==============================================================================

--------------------------------------------------------------------------------
--------------------------------------------------------------------------------

local function Undo()
    if #undostack > 0 then
        redostack[#redostack+1] = currentState
        currentState = table.remove(undostack)
        RestoreState( currentState )
        f.drawgui()
    end
--   showButton(mi.undo)
end
--------------------------------------------------------------------------------

local function Redo()
    if #redostack > 0 then
        undostack[#undostack+1] = currentState
        currentState = table.remove(redostack) 
        RestoreState( currentState )
        f.drawgui()
    end
--   showButton(mi.redo)
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------

local function doClear()
   local rect = g.getrect()
   if (gr.doClear()) then
      f.checkStateChg()
   end
end
--------------------------------------------------------------------------------

local function doCopy()
   g.show("doCopy")
   local rect = g.getrect()
   if (gr.doCopy()) then
--      f.checkStateChg()	-- cannot undo or redo copy
   else
      util.note("nothing to copy")
   end
end

--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
local function makeListLines(str)
   local list = {}
   for word in string.gmatch(str, "([^\n]*)\n?") do
      list[#list + 1] = word
   end
   return list
end
--------------------------------------------------------------------------------

local function doPaste()
   g.show("doPaste")
   if (gr.doPaste()) then
--      f.checkStateChg()	-- cannot undo or redo paste
   end
end
--------------------------------------------------------------------------------

local function rub()
   local width =   tonumber(m_Item.value[mi.width])
   local hight =   tonumber(m_Item.value[mi.hight])
   if (width == 0) then width = hight end
   if (hight == 0) then hight = width end
   local pat = g.getclipstr()
   local x = tonumber(string.match(pat, '[xX] *= *([0-9\n]*)'))
   local y = tonumber(string.match(pat, '[yY] *= *([0-9\n]*)'))
   if (x <= width) and (y <= hight) then
      pat = pat:match('(.*y[^,\n]*)')..'\n'..pat:match('[^\n]*\n(.*)')
      local pType = pat:match('([oOaA])'):upper()
      if ((pType == 'O') and gr.threeState) then
         g.setclipstr(pat:gsub('[oO]','A'):gsub('b','B'))
      elseif (( pType == 'A') and not gr.threeState) then
         g.setclipstr(pat:gsub('[aA]','o'):gsub('B','b'))
      end
      g.paste(-(width//2), -(hight//2), "copy")
   else
      util.note("Pattern size "..x..','..y.." too big")
   end
   g.update()
end
--------------------------------------------------------------------------------

local function doRotateRight()
   for i = 1, tonumber(m_Item.value[mi.rotateStep]) do
      gr.doRotateRL('r')
   end
   f.checkStateChg()
end
--------------------------------------------------------------------------------

local function doRotateLeft()
   for i = 1, tonumber(m_Item.value[mi.rotateStep]) do
      gr.doRotateRL('l')
   end
   f.checkStateChg()
end
--------------------------------------------------------------------------------

local function doRotateUp()
   for i = 1, tonumber(m_Item.value[mi.rotateStep]) do
      gr.doRotateUD('u')
   end
   f.checkStateChg()
end
--------------------------------------------------------------------------------

local function doRotateDown()
   for i = 1, tonumber(m_Item.value[mi.rotateStep]) do
      gr.doRotateUD('d')
   end
   f.checkStateChg()
end
--------------------------------------------------------------------------------

local function doRotateStep()
   local oldVal = m_Item.value[mi.rotateStep]
   if (getText(mi.rotateStep, 'Rotate Step Size ', typeSeedList)) then 
      reSetMbarButtonItem(mi.rotateStep)
      f.checkStateChg(mi.rotateStep)
   end
end
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
--==============================================================================
--------------------------------------------------------------------------------

function util.newLog(newlogFile)
   logFile = newlogFile
   gr.init(newlogFile, util.report, seed, geo)	--#t#seed,geo
   gf.init(newlogFile, util, gr, gs)
   gs.init(newlogFile)
   gsCBrk.init(newlogFile, "default")
end
--------------------------------------------------------------------------------

function util.oldLog()
   util.newLog(io.open ( logFileName , "a"))
   return logFile
end



--------------------------------------------------------------------------------

local function CreateMenuBar()
    -- create the menu bar and add some menus
    local item = 1
    mbar = op.menubar()

    -- add items to File menu
    mbar.addmenu("File")
    item = addMbarItem(menuFile,       1, mi.loadScript)
    item = addMbarItem(menuFile,    item, mi.pasteScript)
    item = addMbarItem(menuFile,    item, mi.sciptRun)
    item = addMbarItem(menuFile,    item, mi.ruleDir)
    item = addMbarItem(menuFile,    item, mi.exit)

    -- add items to Edit menu
    mbar.addmenu("Edit")
    item = addMbarItem(menuEdit,       1, mi.undo)
    item = addMbarItem(menuEdit,    item, mi.redo)
    item = addMbarLine  (menuEdit,    item)
    item = addMbarItem(menuEdit,    item, mi.clear)
    item = addMbarItem(menuEdit,    item, mi.copyPat)
    item = addMbarItem(menuEdit,    item, mi.pastePat)
    item = addMbarItem(menuEdit,    item, mi.rotateR)
    item = addMbarItem(menuEdit,    item, mi.rotateL)
    item = addMbarItem(menuEdit,    item, mi.rotateU)
    item = addMbarItem(menuEdit,    item, mi.rotateD)
    item = addMbarItem(menuEdit,    item, mi.rotateStep)
 
    -- add items to Universe menu
    mbar.addmenu("Universe")
    item = addMbarItem(menuUniverse,    1, mi.step)
    item = addMbarItem(menuUniverse, item, mi.run)
    item = addMbarItem(menuUniverse, item, mi.runN)
    item = addMbarItem(menuUniverse, item, mi.run2G)
    item = addMbarLine  (menuUniverse, item)
    item = addMbarItem(menuUniverse, item, mi.swapComp)
    item = addMbarItem(menuUniverse, item, mi.build)
    item = addMbarLine  (menuUniverse, item)
    item = addMbarItem(menuUniverse, item, mi.rule)
    item = addMbarItem(menuUniverse, item, mi.width)
    item = addMbarItem(menuUniverse, item, mi.hight)
    item = addMbarItem(menuUniverse, item, mi.seed)
    item = addMbarItem(menuUniverse, item, mi.geo)
    item = addMbarItem(menuUniverse, item, mi.geoSize)
    item = addMbarItem(menuUniverse, item, mi.randomPC)
    item = addMbarItem(menuUniverse, item, mi.isleType)
    item = addMbarItem(menuUniverse, item, mi.invertIsle)
    
    -- add items to List Menu
    mbar.addmenu("Lists");item = 1
    item = addMbarItem(menuLists,  item,    mi.nextRule)
    item = addMbarItem(menuLists,  item,    mi.ruleList)
    item = addMbarItem(menuLists,  item,    mi.ruleListNo)
    item = addMbarLine  (menuLists,  item)
    item = addMbarItem(menuLists,  item,    mi.nextSeed)
    item = addMbarItem(menuLists,  item,    mi.seedList)
    item = addMbarItem(menuLists,  item,    mi.seedListNo)

    -- add items to View menu
    mbar.addmenu("View")
    item = addMbarItem(menuView,  1,    mi.fit)
    item = addMbarItem(menuView,  item, mi.ruleFormat)
    item = addMbarItem(menuView,  item, mi.ruleDesc)
    item = addMbarItem(menuView,  item, mi.sciptList)
    item = addMbarItem(menuView,  item, mi.islandResults)
    item = addMbarLine  (menuView,  item)
    item = addMbarItem(menuView,  item, mi.help)

    -- add items to Layer menu
    mbar.addmenu("Layer")
    item = addMbarItem(menuLayer,  1,       mi.layStack)
    item = addMbarItem(menuLayer,  item,    mi.layTile)
    item = addMbarItem(menuLayer,  item,    mi.SetLayer)
    item = addMbarItem(menuLayer,  item,    mi.layDelExtra)

    -- add items to Tools menu
    mbar.addmenu("Tools")
    item = addMbarItem(menuTools,  1,    mi.findFate)
    item = addMbarItem(menuTools,  item, mi.findFateType)
    item = addMbarItem(menuTools,  item, mi.FEstep1)
    item = addMbarItem(menuTools,  item, mi.maxGen)
    item = addMbarItem(menuTools,  item, mi.FEminOsc)
    item = addMbarItem(menuTools,  item, mi.FEmaxOsc)
    item = addMbarLine  (menuTools,  item)
    item = addMbarItem(menuTools,  item, mi.islandAnalysis)
    item = addMbarItem(menuTools,  item, mi.islandResults)
    
    -- add items to Help menu    
    mbar.addmenu("Help")
    item = addMbarItem(menuHelp,  1,    mi.help)
    item = addMbarItem(menuHelp,  item, mi.helpAB)
end
--------------------------------------------------------------------------------

function f.EndableMBar()
    enableMbarButtonItem(mi.sciptList,        gf.ScriptLoaded())
    enableMbarButtonItem(mi.sciptRun,         gf.ScriptLoaded())
    enableMbarButtonItem(mi.undo,             #undostack > 0)
    enableMbarButtonItem(mi.redo,             #redostack > 0)
    enableMbarButtonItem(mi.geoSize,          m_Item.value[mi.geo]:sub(1,1) == "I")
    enableMbarButtonItem(mi.invertIsle,       m_Item.value[mi.geo]:sub(1,1) == "I")
    enableMbarButtonItem(mi.islandResults,    mi.islandResultsValid)
    enableMbarButtonItem(mi.layDelExtra,      g.numlayers() > 1)
    enableMbarButtonItem(mi.SetLayer,         g.numlayers() > 1)
    enableMbarButtonItem(mi.nextRule,         m_Item.value[mi.nextRule] ~= "*")
    m_Item.button[mi.nextRuleTB].enable(      m_Item.value[mi.nextRule] ~= "*")
    enableMbarButtonItem(mi.nextSeed,         m_Item.value[mi.nextSeed] ~= "*")
    m_Item.button[mi.nextSeedTB].enable(      m_Item.value[mi.nextSeed] ~= "*")
    mbar.show(0, 0, ovwd, mbarht)
end
--------------------------------------------------------------------------------

local function nullAction()
   g.show("nullAction")
end
--------------------------------------------------------------------------------

local function CreatePopUpMenus()  
    -- text in pop-up menus is shadowed
    op.textshadowx = 2
    op.textshadowy = 2

    -- create a pop-up menu for Find Fate Parameters
    ffTypeMenu = op.popupmenu()
    ffTypeMenu.additem(" OSC",setFFosc)
    ffTypeMenu.additem(" Break", setFFbreak)

    -- create a pop-up menu for Island(geo) type
    isleTypeMenu = op.popupmenu()
    isleTypeMenu.additem(" Random", setIsleTypeRand)
    isleTypeMenu.additem(" Solid", setIsleTypeSolid)
    
    -- create a pop-up menu for geo
    geoMenu = op.popupmenu()
    geoMenu.additem("Orthogonal", setGeoOrthog)
    geoMenu.additem("Diagonal",setGeoDiag)
    geoMenu.additem("Random", setGeoRand)
    geoMenu.additem("Random_Diagonal",setGeoRandDiag)
    geoMenu.additem("Checkerboard", setGeoCB)
    geoMenu.additem("Round Island", setGeoIsleR)
    geoMenu.additem("Diamond Island", setGeoIsleD)
    geoMenu.additem("Square Island", setGeoIsleS)
end

--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
local function createoverlay()
    -- create overlay for gui
    
    --   File     Load Save Options Help x
    --   Pattern  Geometry  Width NNNN Hight NNNN Seed NNNN Build
    --   Rule   format  NNNN  Rule Descripton
    --   RuleList Next NNNNNNNNNNNNNNNNNNNNNNNNNNNNN
    --   Action Step  Back   Run findFate  Swap:comp Reset
    --   Options ?
    
    ovwd, ovht = viewwd, viewht
    
    ov("create "..ovwd.." "..ovht)
    overlaycreated = true
    op.buttonht = 20
    op.textgap = 8

    setfont()

    -- create Menu bar items-----------------------------------------------------
    -- File Menu / buttons
    mi.loadScript     = newMbarItem("Load Script from File",           "",      doScriptLoad)
    mi.pasteScript    = newMbarItem("Paste Script",                    "",      doScriptPaste)
    mi.sciptRun       = newMbarItem("Run Script",                      "",      doScriptRun)
    mi.ruleDir        = newMbarItem("Set Rule Folder ",                "SymCACP-1/Rules/",doRulesDir)
    mi.exit           = newMbarItem("Exit",                            "",      exitbrowser)
    
    -- Edit Menu 
    mi.undo           = newMbarItem("Undo",                            "",      Undo)
    mi.redo           = newMbarItem("Redo",                            "",      Redo)
    mi.clear          = newMbarItem("Clear",                           "",      doClear)
    mi.copyPat        = newMbarItem("Copy Pattern",                    "",      doCopy)
    mi.pastePat       = newMbarItem("Paste Pattern",                   "",      doPaste)
    mi.rotateR        = newMbarItem("Rotate Right",                    "",      doRotateRight)
    mi.rotateL        = newMbarItem("Rotate Left",                     "",      doRotateLeft)
    mi.rotateU        = newMbarItem("Rotate Up",                       "",      doRotateUp)
    mi.rotateD        = newMbarItem("Rotate Down",                     "",      doRotateDown)
    mi.rotateStep     = newMbarItem("Rotate Step :",                   "1",     doRotateStep)

    -- Universe Menu    
    mi.step           = newMbarItem("Step",                            "",       doStep)
    mi.run            = newMbarItem("Run",                             "",       doRun)
    mi.run2G          = newMbarItem("Run to Generation :",             "500",    dorun2G)
    mi.runN           = newMbarItem("Run n Generations :",             "500",    doRunN)
    mi.swapComp       = newMbarItem("Swap:Comp",                       "",       doSwapComp)
    mi.build          = newMbarItem("Rebuild",                           "",       doBuild)
    mi.rule           = newMbarItem("Change Rule :",                   "H1D8",   doTextRule)
    mi.width          = newMbarItem("Change Width: ",                  "50",     doTextWidth)
    mi.widthLAB       = newMbarItem("Width",                           "Width",  doNoop)
    mi.hight          = newMbarItem("Change Hight: ",                  "50",     doTextHight)
    mi.seed           = newMbarItem("Change Seed: ",                   "10102",  doTextSeed)
    mi.seedLAB        = newMbarItem("Seed",                            "Seed",   doNoop)
    mi.geo            = newMbarItem("Change Geometry: ",               "Random", doGeoMenu)
    mi.geoSize        = newMbarItem("Change Island Size: ",            "0",      doGeoSize)
    mi.isleType       = newMbarItem("Change Island Type: ",            "Random", doIsleType)
    mi.invertIsle     = newMbarItem("Invert Random Island: ",          "false",  doInvertIsle)
    mi.randomPC       = newMbarItem("Random Percentage: ",             "50",     doRandomPC)

    -- List Menu
    mi.nextRule       = newMbarItem("Rule List Build item: ",          "",       doBldnextRule)
    mi.nextRuleTB     = newMbarItem("Next Rule ",                      "",       doBldnextRule)
    mi.ruleList       = newMbarItem("Change Rule List :",              "H000",   doTextRuleList)
    mi.ruleListNo     = newMbarItem("Change next Rule item: ",         "1",      doChgRuleListNo)

    mi.nextSeed       = newMbarItem("Seed List Build item: ",          "",       doBldnextSeed)
    mi.nextSeedTB     = newMbarItem("Next Seed ",                      "",       doBldnextSeed)
    mi.seedList       = newMbarItem("Change Seed List :",              "010,102",doTextSeedList)
    mi.seedListNo     = newMbarItem("Change next Seed item: ",         "1",      doChgSeedListNo)
    
    -- View Menu
    mi.fit            = newMbarItem("Fit",                             "",       f.doFit)
    mi.ruleFormat     = newMbarItem("Change Rule Format :",            "Hex  ",  doRuleFormat)
    mi.ruleDesc       = newMbarItem("Rule Description :",              spaces(100), doRuleDesc)
    mi.sciptList      = newMbarItem("List Script",                     "",       doScriptList)
 
    -- Layer Menu
    mi.layStack       = newMbarItem("Stack Layers ",                   "False",  doLayer_Stack)
    mi.layTile        = newMbarItem("Tile Layers ",                    "False",  doLayer_Tile)
    mi.SetLayer       = newMbarItem("Change Layer :",                  "1",      doLayer_Set)
    mi.layDelExtra    = newMbarItem("Delete Extra Layers",             "",       doLayer_delEx)
    
    -- tools menu    
    mi.findFate        = newMbarItem("Find Fate",                       "",       doFindFate)
    mi.findFateType    = newMbarItem("Change Find Fate Type: ",         "",       doFEopsMenu)
    mi.FEstep1        = newMbarItem("Change Find Fate Initial Step: ", "15",     doTextStep1Gen)
    mi.maxGen         = newMbarItem("Change Find Fate Max Generation:","5000",   doTextmaxGen)
    mi.FEminOsc       = newMbarItem("Change Find Fate Min Period: ",   "1",      doTextMinOsc)
    mi.FEmaxOsc       = newMbarItem("Change Find Fate Max Period: ",   "100",    doTextMaxOsc)
    mi.islandAnalysis = newMbarItem("Island Analysis",                 "",       doIsland)
    mi.islandResults  = newMbarItem("Island Analysis Results",         "",       doislandResults)
    
    -- Help Menu
    mi.help           = newMbarItem("Help",                            "",       showhelp)
    mi.helpAB         = newMbarItem("About",                           "",       showhelpAB);

    -- dummy item for description file name    
    mi.ruleDescFile   = newMbarItem("ruleDesc File",                   "ruleDesc.txt",nil)
    
    -- extra item number for checking generation number changes
    mi.gen           = newMbarItem("generation",                       "",       nil)
-- create buttons
    op.textshadowx = 2
    op.textshadowy = 2
    local adjbmht = 50
    local y = floor((guiht - op.buttonht+adjbmht) / 2)
    
    local tbWD = gapx
    local menuNo = 1
    
    tbWD = makeButton(mi.step,       tbWD, y, menuNo, m_Item.text[mi.step])
    tbWD = makeButton(mi.undo,       tbWD, y, menuNo, m_Item.text[mi.undo])
    tbWD = makeButton(mi.redo,       tbWD, y, menuNo, m_Item.text[mi.redo])
    tbWD = makeButton(mi.run,        tbWD, y, menuNo, m_Item.text[mi.run])
    tbWD = makeButton(mi.swapComp,   tbWD, y, menuNo, m_Item.text[mi.swapComp])
    tbWD = makeButton(mi.nextRuleTB, tbWD, y, menuNo, "Next Rule")
    tbWD = makeButton(mi.nextSeedTB, tbWD, y, menuNo, "Next Seed")
    tbWD = makeButton(mi.help,       tbWD, y, menuNo, m_Item.text[mi.help])
    tbWD = makeButton(mi.exit,       tbWD, y, menuNo, m_Item.text[mi.exit])

    local gapH = gapy + m_Item.button[mi.step].ht
    tbWD = gapx
    menuNo = 2
    y = y + gapH
    
    tbWD = makeButton(mi.rule,           tbWD, y, menuNo, "H1D8        ")
    tbWD = makeButton(mi.widthLAB,       tbWD, y, menuNo, "Width")
    m_Item.textOnly[mi.widthLAB] = true
    m_Item.button[mi.widthLAB].enable(false)
    tbWD = makeButton(mi.width,          tbWD, y, menuNo, "10        ")
    tbWD = makeButton(mi.seedLAB,        tbWD, y, menuNo, "Seed")
    m_Item.textOnly[mi.seedLAB] = true
    m_Item.button[mi.seedLAB].enable(false)
    tbWD = makeButton(mi.seed,           tbWD, y, menuNo, "10        ")
    tbWD = makeButton(mi.ruleDesc,       tbWD, y, menuNo, spaces(100))
    g.show("made mi.ruleDesc")
    m_Item.maxLabel[mi.ruleDesc] = 62
    
    tbWD = gapx
--    menuNo = 4
--    y = y + gapH
        
--    tbWD = makeButton(mi.loadScript, tbWD, y, menuNo, "Load Script")
--    tbWD = makeButton(mi.sciptList,  tbWD, y, menuNo, "List Script")
--    tbWD = makeButton(mi.sciptRun,   tbWD, y, menuNo, "Run Script")
--    tbWD = gapx
    
    -- create the clip for clipping info text
    tBarht  = (menuNo+1) * gapH +5
    ov("create "..ovwd.." "..ovht.." "..textrect)    
end
--------------------------------------------------------------------------------

local function checkforresize()
    local newwd, newht = g.getview(g.getlayer())
    if newwd ~= viewwd or newht ~= viewht then
        -- hide controls so show draws correct background
        for bi =1, m_Items do
           if(m_Item.button[bi]) then
              m_Item.button[bi].hide()
           end
        end
        
        -- resize overlay
        if newwd < 1 then newwd = 1 end
        if newht < 1 then newht = 1 end
        viewwd = newwd
        ovwd = newwd
        viewht = newht
        ov("resize "..viewwd.." "..viewht)
        f.drawgui() 
    end
end
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------

local function SaveGollyState()
    local oldstate = {}
    oldstate.scroll = g.setoption("showscrollbars", 0)
    oldstate.time = g.setoption("showtimeline", 0)
    oldstate.tool = g.setoption("showtoolbar", 0)
    oldstate.status = g.setoption("showstatusbar", 1) --#t#
    oldstate.layer = g.setoption("showlayerbar", 0)
    oldstate.edit = g.setoption("showeditbar", 0)
    oldstate.tile = g.setoption("tilelayers", 0)
    oldstate.stack = g.setoption("stacklayers", 0)
    oldstate.files = g.setoption("showfiles", 0)
    oldstate.filesdir = g.getdir("files")
    return oldstate
end

--------------------------------------------------------------------------------

local function RestoreGollyState(oldstate)
    if overlaycreated then
       ov("delete")
       overlaycreated = false
    end
    g.setoption("showscrollbars", oldstate.scroll)
    g.setoption("showtimeline", oldstate.time)
    g.setoption("showtoolbar", oldstate.tool)
    g.setoption("showstatusbar", oldstate.status)
    g.setoption("showlayerbar", oldstate.layer)
    g.setoption("showeditbar", oldstate.edit)
    g.setoption("tilelayers", oldstate.tile)
    g.setoption("stacklayers", oldstate.stack)
    g.setoption("showfiles", oldstate.files)
    g.setdir("files", oldstate.filesdir)
end

--==============================================================================

local function contollLoop()

    -- loop until escape pressed or exit clicked
    exitnow = false
    while not exitnow do

       g.sleep(5)      -- don't hog the CPU when idle

       -- check for window reisze
       checkforresize()
       
       -- decode key presses
       -- check for event
       local event = op.process(g.getevent())
       if event == "key r none" then
          doRun()
       elseif event == "key s none" then
          doStep()
       elseif event == "key b none" then
          Undo()
       elseif event == "key f none" then
          Redo()
       elseif event == "key equal shift" then
          g.reset()
       elseif event == "key ? none" then
          showhelp()
       else
       -- pass event to Golly for processing
          g.doevent(event)
       end
       
       if event and #event > 0 then
          f.drawgui()
       end
        	
    end
end
--------------------------------------------------------------------------------
--==============================================================================
-- Main
--==============================================================================
oldstate = SaveGollyState()
util.report = reporter()
gr.init(logFile, util.report, seed, geo)
gf.init(logFile, util, gr, gs)
gs.init(logFile)
gsCBrk.init(logFile, "default")
createoverlay()


loadsettings()
CreateMenuBar()
      
m_Item.value[mi.nextRule] = m_Item.value[mi.ruleListNo]
reSetMbarButtonItem(mi.nextRule)
m_Item.value[mi.nextSeed] = m_Item.value[mi.seedListNo]
reSetMbarButtonItem(mi.nextSeed)
CreatePopUpMenus()
loadDescriptions()
currentState = SaveState()
reset_StackTile("no Note")
f.drawgui()

g.show("GlobalSwitch_ChaosSearch "..util.bool2txt(GlobalSwitch_ChaosSearch))

local status, err = xpcall(contollLoop, gp.trace)

if err then g.continue(err) end

-- this code is always executed, even after escape/error;
-- clear message line in case there was no escape/error
gf.saveResults("default")
RestoreGollyState(oldstate)

savesettings()
g.check(false)
--g.show("")
logFile:write("\nEnd of Log\n")
logFile:close()
if overlaycreated then
   ov("delete")
end

--------------------------------------------------------------------
--                      	END OF FILE
--------------------------------------------------------------------
